<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Branch;
use App\Models\Form;
use App\Models\Project;
use App\Models\ProjectForm;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Maatwebsite\Excel\Facades\Excel;

class ProjectController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $sort_search = null;
        $projects = Project::orderBy('created_at', 'desc');
        $whereArr = [];
        if ($request->has('search') && !empty($request->search)) {
            $sort_search = $request->search;
            $whereArr[] = ['name', 'like', '%' . $sort_search . '%'];
            // $projects = $projects->where('name', 'like', '%' . $sort_search . '%')->orWhere('email', 'like', '%' . $sort_search . '%');
        }
        if ($request->has('branch_id') && $request->branch_id != 0) {
            $whereArr[] = ['branch_id', $request->branch_id];
            // $projects  = $projects->where('branch_id', $request->branch_id);
            $branch_id = $request->branch_id;
        } else {
            $branch_id = null;
        }


        $projects = $projects->where($whereArr)->paginate(20);
        $branches = Branch::get();
        return view('admin.project.index', compact('projects', 'sort_search', 'branches', 'branch_id'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $branches = Branch::pluck('name', 'id');
        $periodicForms = Form::where('periodic_onetime', 'periodic')->pluck('name', 'id');
        return view('admin.project.create', compact('branches', 'periodicForms'));
    }

    public function getProjectsByBranch(Request $request)
    {
        $projects = Project::where('branch_id', $request->branch_id)->pluck('name', 'id')->prepend(trans('messages.All Projects'), 0);
        $branch = Branch::where('id', $request->branch_id)->first();
        return view('admin.project.partial.get_projects_by_branch', compact('projects', 'branch'));
    }


    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // return $request;
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
             'email' => ['nullable', 'string', 'email', 'max:255'],
             'password' => ['required', 'string', 'min:8', 'confirmed'],
            'phone' => ['nullable', 'numeric']
        ]);
        // return $request; $form->projects()->sync($lastProjects_ids);
        $project = new Project;
        $project->name = $request->name;
        $project->branch_id = $request->branch_id;
        $project->hse_manager_name = $request->hse_manager_name;
        $project->email = $request->email;
        $project->phone = $request->phone;
        $project->password = Hash::make($request->password);
        $project->save();
        $project->forms()->sync($request->forms);
        session()->flash('notif', trans('messages.Added successfully.'));
        return redirect()->route('project.index');
    }

    public function uploadProjectExcelFile(Request $request)
    {
        $file = $request->file('file');
        $data = Excel::toArray([], $file)[0];
        foreach ($data as $key => $value) {
            if ($key != 0) {
                $branch =  Branch::firstOrCreate(
                    ['name' =>  $value[0]]
                );
                Project::create([
                    'name' =>  $value[4],
                    'branch_id' => $branch->id,
                    'hse_manager_name' =>  $value[1],
                    'email' =>  $value[2],
                    'password' => Hash::make($value[3])
                ]);
            }
        }
        return response()->json(['success' => true]);
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Project $project, Request $request)
    {
        $sort_search = null;
        $forms = ProjectForm::where('project_id', $project->id)->orderBy('created_at', 'desc');
        if ($request->has('search')) {
            $sort_search = $request->search;
            $forms = $forms->where('serial_number', 'like', '%' . $sort_search . '%');
        }
        if ($request->has('status') && $request->status != 'all') {
            $forms = $forms->where('status', $request->status);
            $status = $request->status;
        } else {
            $status = null;
        }
        if ($request->has('form_id') && $request->form_id != 0) {
            $forms = $forms->where('form_id', $request->form_id);
            $form_id = $request->form_id;
        } else {
            $form_id = $request->form_id;
        }
        $forms = $forms->paginate(15);
        $branches = Branch::get();
        $originalForms = Form::pluck('name', 'id')->prepend(trans('messages.All Forms'), 0);
        return view('admin.project.show', compact('project', 'originalForms', 'forms', 'status', 'form_id', 'branches', 'sort_search'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Project $project)
    {
        $branches = Branch::pluck('name', 'id');
        $periodicForms = Form::where('periodic_onetime', 'periodic')->pluck('name', 'id');
        return view('admin.project.edit', compact('project', 'branches', 'periodicForms'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Project $project)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            // 'email' => Rule::unique('projects', 'email')->ignore($project->id),
            'phone' => ['nullable', 'numeric']
        ]);
        $project->name = $request->name;
        $project->branch_id = $request->branch_id;
        $project->hse_manager_name = $request->hse_manager_name;
        $project->email = $request->email;
        $project->phone = $request->phone;
        $project->save();
        $project->forms()->sync($request->forms);
        session()->flash('notif', trans('messages.Edited successfully.'));
        return redirect()->route('project.index');
    }

    public function viewPassword($id)
    {
        $project = Project::find($id);
        if(!empty($project))
            return view('admin.project.password', compact('project'));


        return  redirect()->back();


    }

    public function password(Request $request, $id)
    {

        $request->validate([
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);
        $project = Project::find($id);
        if(!empty($project))
        {
            $project->password = Hash::make($request->password);
            $project->save();
            return redirect()->route('project.index');
        }
        return  redirect()->back();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Project $project)
    {
        $project->delete();
        session()->flash('notif', trans('messages.Project deleted successfully'));
        return redirect()->route('project.index');
    }

    public function getProjectsByBranchId(Request $request)
    {
        if ($request->branch_id != 0) {
            $projects =  Project::where('branch_id', $request->branch_id)->pluck('name', 'id')->prepend(trans('messages.All Projects'), 0);
        } else {
            $projects = Project::pluck('name', 'id')->prepend(trans('messages.All Projects'), 0);
        }
        $project_id = $request->project_id;
        return view('admin.project.partial.get_projects_by_branchid', compact('projects', 'project_id'));
    }
}
