<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Branch;
use App\Models\Project;
use App\Models\ProjectForm;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rule;
use Maatwebsite\Excel\Facades\Excel;

class ProjectController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $sort_search = null;
        $projects = Project::orderBy('created_at', 'desc');
        if ($request->has('search')) {
            $sort_search = $request->search;
            $projects = $projects->where('name', 'like', '%' . $sort_search . '%');
        }
        $projects = $projects->paginate(20);
        return view('admin.project.index', compact('projects', 'sort_search'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $branches = Branch::pluck('name', 'id');
        return view('admin.project.create', compact('branches'));
    }

    public function getProjectsByBranch(Request $request)
    {
        $projects = Project::where('branch_id', $request->branch_id)->pluck('name', 'id')->prepend(trans('messages.All Projects'), 0);
        $branch = Branch::where('id', $request->branch_id)->first();
        return view('admin.project.partial.get_projects_by_branch', compact('projects', 'branch'));
    }


    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // return $request;
        $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'email' => ['required', 'string', 'email', 'max:255', 'unique:users'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);
        // return $request;
        Project::create([
            'name' => $request->name,
            'branch_id' => $request->branch_id,
            'hse_manager_name' => $request->hse_manager_name,
            'email' => $request->email,
            'password' => Hash::make($request->password)
        ]);
        session()->flash('notif', trans('messages.Added successfully.'));
        return redirect()->route('project.index');
    }

    public function uploadProjectExcelFile(Request $request)
    {
        $file = $request->file('file');
        $data = Excel::toArray([], $file)[0];
        foreach ($data as $key => $value) {
            if ($key != 0) {
               $branch =  Branch::firstOrCreate(
                    ['name' =>  $value[0]]
                );
                Project::create([
                    'name' =>  $value[4],
                    'branch_id' => $branch->id,
                    'hse_manager_name' =>  $value[1],
                    'email' =>  $value[2],
                    'password' => Hash::make( $value[3])
                ]);
            }
        }
        return response()->json(['success' => true]);
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show(Project $project)
    {
        $forms = ProjectForm::where('project_id', $project->id)->orderBy('created_at', 'desc')->get();
        return view('admin.project.show', compact('project', 'forms'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit(Project $project)
    {
        $branches = Branch::pluck('name', 'id');
        return view('admin.project.edit', compact('project', 'branches'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Project $project)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => Rule::unique('projects', 'email')->ignore($project->id),
            'old_password' => 'nullable|string|min:8',
            'new_password' => 'nullable|string|min:8',
        ]);
        if ($request->new_password) {
            if (Hash::check($request->old_password, $project->password)) {
                $project->password =  Hash::make($request->new_password);
            } else {
                session()->flash('notif', 'messages.Password does not match');
                return redirect()->route('project.edit', $project->id);
            }
        }
        $project->name = $request->name;
        $project->hse_manager_name = $request->hse_manager_name;
        $project->email = $request->email;
        $project->save();
        session()->flash('notif', trans('messages.Edited successfully.'));
        return redirect()->route('project.index');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy(Project $project)
    {
        $project->delete();
        session()->flash('notif', trans('messages.Project deleted successfully'));
        return redirect()->route('project.index');
    }
}
