<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;

use App\Models\Branch;
use App\Http\Requests\StoreBranchRequest;
use App\Http\Requests\UpdateBranchRequest;
use App\Models\Project;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Maatwebsite\Excel\Facades\Excel;

class BranchController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        $sort_search = null;
        $branches = Branch::orderBy('created_at', 'desc');
        if ($request->has('search')) {
            $sort_search = $request->search;
            $branches = $branches->where('name', 'like', '%' . $sort_search . '%');
        }
        $branches = $branches->paginate(20);
        return view('admin.branch.index', compact('branches', 'sort_search'));
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('admin.branch.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \App\Http\Requests\StoreBranchRequest  $request
     * @return \Illuminate\Http\Response
     */
    public function store(StoreBranchRequest $request)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255','unique:branches,name'],
            'email' => ['nullable', 'string', 'email', 'max:255'],
            'password' => ['required', 'string', 'min:8', 'confirmed'],
            'phone' => ['nullable', 'numeric']
        ]);
        $branch = new Branch;
        $branch->name = $request->name;
        $branch->email = $request->email;
        $branch->phone = $request->phone;
        $branch->user_name = $request->user_name;
        $branch->password = Hash::make($request->password);
        $branch->save();
        return redirect()->route('branch.index');
    }

    public function uploadBranchExcelFile(Request $request)
    {
        $file = $request->file('file');
        $data = Excel::toArray([], $file)[0];
        foreach ($data as $key => $value) {
            if ($key != 0) {
                Branch::firstOrCreate(
                    ['name' =>  $value[0]]
                );
            }
        }
        return response()->json(['success' => true]);
    }

    /**
     * Display the specified resource.
     *
     * @param  \App\Models\Branch  $branch
     * @return \Illuminate\Http\Response
     */
    public function show(Branch $branch)
    {
        $projects = Project::where('branch_id', $branch->id)->orderBy('created_at', 'desc')->get();
        return view('admin.branch.show', compact('branch', 'projects'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  \App\Models\Branch  $branch
     * @return \Illuminate\Http\Response
     */
    public function edit(Branch $branch)
    {
        return view('admin.branch.edit', compact('branch'));
    }

    public function viewPassword($id)
    {
        $branch = Branch::find($id);
        if(!empty($branch))
            return view('admin.branch.password', compact('branch'));


        return  redirect()->back();


    }

    public function password(Request $request, $id)
    {
        $request->validate([
            'password' => ['required', 'string', 'min:8', 'confirmed'],
        ]);
        $branch = Branch::find($id);
        if(!empty($branch))
        {
            $branch->password = Hash::make($request->password);
            $branch->save();
            return redirect()->route('branch.index');
        }
        return  redirect()->back();
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \App\Http\Requests\UpdateBranchRequest  $request
     * @param  \App\Models\Branch  $branch
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateBranchRequest $request, Branch $branch)
    {

        $request->validate([
            'name' => ['required', 'string', 'max:255','unique:branches,name,'.$branch->id],
            'email' => ['nullable', 'string', 'email', 'max:255'],
            'phone' => ['nullable', 'numeric']
        ]);
        $branch->name = $request->name;
        $branch->email = $request->email;
        $branch->phone = $request->phone;
        $branch->user_name = $request->user_name;
        $branch->save();
        return redirect()->route('branch.index');
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  \App\Models\Branch  $branch
     * @return \Illuminate\Http\Response
     */
    public function destroy(Branch $branch)
    {
        $branch->delete();
        session()->flash('notif', trans('messages.Branch deleted successfully'));
        return redirect()->route('branch.index');
    }
}
